/*****************************************************************************
**
**  ctl::to_string version 2.100
**
**  Copyright (c) 2018-2024, Nozomu Katoo. All rights reserved.
**
**  Redistribution and use in source and binary forms, with or without
**  modification, are permitted provided that the following conditions are
**  met:
**
**  1. Redistributions of source code must retain the above copyright notice,
**     this list of conditions and the following disclaimer.
**
**  2. Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in the
**     documentation and/or other materials provided with the distribution.
**
**  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
**  IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
**  THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
**  PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
**  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
**  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
**  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
**  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
**  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
**  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
**  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
******************************************************************************
**/

#ifndef CTL_STRING_TEMPLATE_LIBRARY
#define CTL_STRING_TEMPLATE_LIBRARY

#include <cstring>
#include <limits>
#include <algorithm>
#include <utility>

#if !defined(NAMESPACE_CTLSTRING)
#define NAMESPACE_CTLSTRING ctl
#endif

namespace NAMESPACE_CTLSTRING
{
	namespace string_internal_
	{

template <typename T>
struct casetables
{
	static const T lc[];
	static const T uc[];
};
template <typename T>
const T casetables<T>::lc[] =
{
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,	//  "0123456789";
	0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,	//  "abcdefghij"
	0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74,	//  "klmnopqrst"
	0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x2e, 0x2b,	//  "uvwxyz.+"
	0x2d, 0x69, 0x6e, 0x66,	//  "-inf"
	0x2d, 0x71, 0x6e, 0x61, 0x6e,	//  "-qnan"
	0x2d, 0x73, 0x6e, 0x61, 0x6e	//  "-snan"
};
template <typename T>
const T casetables<T>::uc[] =
{
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,	//  "0123456789";
	0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a,	//  "ABCDEFGHIJ"
	0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54,	//  "KLMNOPQRST"
	0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x2e, 0x2b,	//  "UVWXYZ.+"
	0x2d, 0x49, 0x4e, 0x46,	//  "-INF"
	0x2d, 0x51, 0x4e, 0x41, 0x4e,	//  "-QNAN"
	0x2d, 0x53, 0x4e, 0x41, 0x4e	//  "-SNAN"
};

template <>
const char casetables<char>::lc[] = "0123456789abcdefghijklmnopqrstuvwxyz.+-inf-qnan-snan";
template <>
const char casetables<char>::uc[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ.+-INF-QNAN-SNAN";

template <>
const wchar_t casetables<wchar_t>::lc[] = L"0123456789abcdefghijklmnopqrstuvwxyz.+-inf-qnan-snan";
template <>
const wchar_t casetables<wchar_t>::uc[] = L"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ.+-INF-QNAN-SNAN";

template <typename stringT>
stringT reverse_(stringT &s)
{
	typedef typename stringT::size_type size_type;
	const size_type mid = s.size() / 2;

	for (size_type i = 0; i < mid; ++i)
		std::swap(s[i], s[s.size() - i - 1]);

	return s;
}

template <typename stringT>
int mul2_(stringT &s, const int radix)
{
	typedef typename stringT::value_type char_type;
	typedef typename stringT::size_type size_type;
	char_type carried = 0;

	for (size_type i = 0; i < s.size(); ++i)
	{
		char_type &ch = s[i] *= 2;

		carried = ((ch = static_cast<char_type>(ch + carried)) >= static_cast<char_type>(radix)) ? (ch = static_cast<char_type>(ch - radix), 1) : 0;
	}
	return carried ? (s.push_back(carried), 1) : 0;
}

template <typename stringT>
int div2_(stringT &s, const int radix)
{
	typedef typename stringT::value_type char_type;
	typedef typename stringT::size_type size_type;
	int borrowed = 0;

	for (size_type i = s.size(); i;)
	{
		char_type &ch = s[--i];

		ch = static_cast<char_type>(ch + (borrowed ? radix : 0));

		borrowed = ch & 1;
		ch /= 2;
	}
	if (borrowed)
		s.insert(0, 1, static_cast<char_type>(radix / 2));

	if (s.size() && s[s.size() - 1] == 0)
	{
		s.resize(s.size() - 1);
		borrowed |= 2;
	}
	return borrowed;
}

template <typename stringT>
int plus_(stringT &s1, const stringT &s2, const int radix)
{
	typedef typename stringT::value_type char_type;
	typedef typename stringT::size_type size_type;
	char_type carried = 0;

	if (s1.size() < s2.size())
		s1.resize(s2.size(), 0);

	for (size_type i = 0; i < s1.size(); ++i)
	{
		char_type &ch = s1[i] += (i < s2.size() ? s2[i] : 0) + carried;
		carried = (ch >= static_cast<char_type>(radix)) ? (ch = static_cast<char_type>(ch - radix), 1) : 0;
	}
	return carried ? (s1.push_back(carried), 1) : 0;
}

template <typename stringT>
int round_(stringT &s, typename stringT::size_type pos, const int radix)
{
	typedef typename stringT::value_type char_type;

	if (pos <= s.size())
	{
		char_type carried = (s[(pos = s.size() - pos)] >= static_cast<char_type>(radix / 2)) ? 1 : 0;

		for (s.erase(0, pos + 1), pos = 0; pos < s.size(); ++pos)
		{
			char_type &ch = s[pos];

			ch = static_cast<char_type>(ch + carried);
			carried = (ch >= static_cast<char_type>(radix)) ? (ch = static_cast<char_type>(ch - radix), 1) : 0;
		}
		if (carried)
			return s.push_back(carried), 1;
	}
	return 0;
}

template <typename T>
static bool is_same_(const T l, const T r)
{
	return std::memcmp(&l, &r, sizeof (T)) == 0;
}

	}	//  namespace string_internal_

template <typename stringT, typename T>
inline stringT to_string(T value, int base = 10, const int precision = 1)
{
	typedef typename stringT::value_type char_type;
	typedef typename stringT::size_type size_type;
	const char_type *const table = (base >= 0 ? string_internal_::casetables<char_type>::lc : (base = 0 - base, string_internal_::casetables<char_type>::uc));
	stringT num;

	if (base >= 2 && base <= 36)
	{
		const bool minus = value < 0 ? (value = 0 - value, true) : false;

		for (; value; value /= base)
		{
			num.push_back(table[value % base]);
		}

		if (precision > 0 && num.size() < static_cast<size_type>(precision))
			num.append(static_cast<size_type>(precision) - num.size(), table[0]);

		if (minus)
			num.push_back(table[38]);
	}
	return string_internal_::reverse_(num);
}

template <typename stringT, typename T>
stringT to_string_fp(T value, int fmt, int precision)
{
	typedef stringT string_type;
	typedef typename stringT::value_type char_type;
	typedef typename stringT::size_type size_type;

	const bool uc = fmt >= 'A' && fmt <= 'Z';

	const char_type *const table = uc ? (fmt += 'a' - 'A', string_internal_::casetables<char_type>::uc) : string_internal_::casetables<char_type>::lc;

	int dppos;

	if ((dppos = 38 + 1, string_internal_::is_same_(value, std::numeric_limits<T>::infinity())) || (--dppos, string_internal_::is_same_(value, -std::numeric_limits<T>::infinity())))
//		return &(uc ? "-INF" : "-inf")[dppos];
		return stringT(&table[dppos], &table[dppos + ((dppos & 1) ? 3 : 4)]);

	if ((dppos = 42 + 1, string_internal_::is_same_(value, std::numeric_limits<T>::quiet_NaN())) || (--dppos, string_internal_::is_same_(value, -std::numeric_limits<T>::quiet_NaN())))
//		return &(uc ? "-QNAN" : "-qnan")[dppos];
		return stringT(&table[dppos], &table[dppos + ((dppos & 1) ? 4 : 5)]);

	if ((dppos = 47 + 1, string_internal_::is_same_(value, std::numeric_limits<T>::signaling_NaN())) || (--dppos, string_internal_::is_same_(value, -std::numeric_limits<T>::signaling_NaN())))
//		return &(uc ? "-SNAN" : "-snan")[dppos];
		return stringT(&table[dppos], &table[dppos + ((dppos & 1) ? 5 : 4)]);

	const int base = fmt == 'a' ? (fmt = 'e', 16) : 10;
	const bool remove0s = (precision < 0) ? (precision = 0 - precision, true) : (fmt == 'g');
	const bool minus = (value < 0 || string_internal_::is_same_(value, static_cast<T>(-0.0))) ? (value = 0 - value, true) : false;
	string_type resstr;

	dppos = 0;
	if (value != 0.0)
	{
		int borrowed = 0;
		T mask = 1.0;
		string_type mstr(1, 1);

		if (value >= mask)
			for (; (value - mask) >= mask; mask *= 2.0)
				dppos += string_internal_::mul2_(mstr, base);
		else
			for (; value < ((dppos -= (string_internal_::div2_(mstr, base) & 2) ? 1 : 0), mask /= 2.0););

		for (int count = 11; value > 0.0 && mask > 0.0 && count < 64; mask /= 2.0)
		{
			if (borrowed)
				resstr.insert(0, 1, 0);

			if (value >= mask)
			{
				value -= mask;
				dppos += string_internal_::plus_(resstr, mstr, base);
			}
			borrowed = string_internal_::div2_(mstr, base) & 1;
		}

		if (base == 16)
		{
			const char_type &lb = resstr[resstr.size() - 1];
			const int shift = (lb & 8) ? 3 : ((lb & 4) ? 2 : ((lb & 2) ? 1 : 0));

			dppos = dppos * 4 + shift;
			if (shift)
			{
				char_type borrowed2 = 0;
				for (size_type i = resstr.size(); i;)
				{
					char_type &ch = resstr[--i] |= borrowed2;
					borrowed2 = ch << 4;
					ch = static_cast<char_type>((ch >> shift) & 15);
				}
				if (borrowed2)
					resstr.insert(0, 1, static_cast<char_type>((borrowed2 >> shift) & 15));
			}
		}
//		dppos += round_(resstr, 18, base);
	}

	if (fmt == 'g')
	{
		dppos += string_internal_::round_(resstr, precision + 1, base);

		if (dppos >= -4 && dppos < precision)
		{
			precision = precision - 1 - dppos;
			dppos < 0 ? (resstr.append(-dppos, 0), dppos = 1) : ++dppos;
			goto STYLE_F;
		}
		goto STYLE_E;
	}

	if (fmt == 'e')
	{
		++precision;
		dppos += string_internal_::round_(resstr, precision + 1, base);

		STYLE_E:

		if (remove0s)
			for (size_type i = 0; precision > 1 && i < resstr.size() && resstr[i] == 0; --precision, ++i);

		if (resstr.size() > static_cast<size_type>(precision))
			resstr.erase(0, resstr.size() - precision);
		else
			resstr.insert(0, static_cast<size_type>(precision) - resstr.size(), 0);

		for (size_type i = 0; i < resstr.size(); ++i)
			resstr[i] = table[static_cast<unsigned int>(resstr[i])];

		if (resstr.size() > 1)
			resstr.insert(resstr.size() - 1, 1, table[36]);

		resstr.insert(0, 1, table[base == 10 ? 14 : 25]);	//  'E''e' 'P''p'
		resstr.insert(0, 1, ((dppos >= 0) ? table[37] : table[38]));

		if (dppos < 0)
			dppos = 0 - dppos;

		const size_type minwidth = base == 10 ? 2 : 1;
		for (size_type inspos = 0; dppos || inspos < minwidth; ++inspos, dppos /= 10)
			resstr.insert(inspos, 1, table[dppos % 10]);

		if (base == 16)
		{
			resstr.push_back(table[33]);	//  'X' 'x'
			resstr.push_back(table[0]);
		}
	}
	else
	{
		dppos < 0 ? (resstr.append(-dppos, 0), dppos = 1) : ++dppos;

		dppos += string_internal_::round_(resstr, precision + dppos + 1, base);

		STYLE_F:

		if (remove0s)
			for (size_type i = 0; precision > 1 && i < resstr.size() && resstr[i] == 0; --precision, ++i);

		precision += dppos;

		if (resstr.size() > static_cast<size_type>(precision))
			resstr.erase(0, resstr.size() - precision);
		else
		{
			const size_type wantedsize = static_cast<size_type>(remove0s ? dppos : precision);
			if (resstr.size() < wantedsize)
				resstr.insert(0, wantedsize - resstr.size(), 0);
		}

		for (size_type i = 0; i < resstr.size(); ++i)
			resstr[i] = table[static_cast<unsigned int>(resstr[i])];

		if (resstr.size() > static_cast<size_type>(dppos))
			resstr.insert(resstr.size() - dppos, 1, table[36]);
	}

	if (minus)
		resstr.push_back(table[38]);

	return string_internal_::reverse_(resstr);
}

template <typename stringT>
inline stringT to_string(double val, const int fmt = 'f', const int precision = 6)
{
	return to_string_fp<stringT>(val, fmt, precision);
}
template <typename stringT>
inline stringT to_string(float val, const int fmt = 'f', const int precision = 6)
{
	return to_string_fp<stringT>(val, fmt, precision);
}
template <typename stringT>
inline stringT to_string(long double val, const int fmt = 'f', const int precision = 6)
{
	return to_string_fp<stringT>(val, fmt, precision);
}

}	//  namespace NAMESPACE_CTLSTRING
#endif	//  CTL_STRING_TEMPLATE_LIBRARY
